<# :
@echo off
cd /d "%~dp0"
:: The following line runs itself as PowerShell (-ExecutionPolicy Bypass to avoid permission errors)
powershell -NoProfile -ExecutionPolicy Bypass -Command "Invoke-Expression ((Get-Content '%~f0') -join [Environment]::NewLine)"
goto :eof
#>

# -------------------------------------------------------
# Please write your PowerShell (.ps1) code below here.
# -------------------------------------------------------

$ErrorActionPreference = "Stop"

# ===============================
# Settings
# ===============================
$VERSION = "Apple SHARP Auto Installer Ver0.1"
$BASE     = "$env:USERPROFILE\SHARP"
$CONDA    = "$BASE\Miniconda"
$ENVNAME  = "sharp"
$PYTHON   = "3.13"

# ===============================
# Create folder
# ===============================
New-Item -ItemType Directory -Force -Path $BASE | Out-Null

# ===============================
# 1. Miniconda Download
# ===============================
Write-Host "$VERSION
This installer uses Miniconda3.
By continuing this installation you are accepting this license agreement:
-------------------------------------------------------------------------
MINICONDA END USER LICENSE AGREEMENT

Copyright Notice: Miniconda(R) (C) 2015, Anaconda, Inc.
All rights reserved. Miniconda(R) is licensed, not sold.
Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer;
2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution;
3. The name Anaconda, Inc. or Miniconda(R) may not be used to endorse or promote products derived from this software without specific prior written permission from Anaconda, Inc.; and
4. Miniconda(R) may not be used to access or allow third parties to access Anaconda package repositories if such use would circumvent paid licensing requirements or is otherwise restricted by the Anaconda Terms of Service.

DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ANACONDA "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE , AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL ANACONDA BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF MINICONDA(R), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
-------------------------------------------------------------------------"
$answer = Read-Host "Do you agree to this LICENSE AGREEMENT? (Y/N)"

if ($answer -match '^[Yy]') {
} else {
	exit 1
}
Write-Host "Miniconda Download (About 100MB)"
$minicondaExe = "$BASE\miniconda.exe"
Invoke-WebRequest `
  -Uri "https://repo.anaconda.com/miniconda/Miniconda3-latest-Windows-x86_64.exe" `
  -OutFile $minicondaExe
# ===============================
# 2. Miniconda Silent Installation
# ===============================
Start-Process $minicondaExe `
  -ArgumentList "/InstallationType=JustMe /AddToPath=0 /RegisterPython=0 /S /D=$CONDA" `
  -Wait

$condaExe = "$CONDA\Scripts\conda.exe"

# ===============================
# 3. SHARP Download & Extract
# ===============================
$sharpZip = "$BASE\sharp.zip"
Invoke-WebRequest `
  -Uri "https://github.com/apple/ml-sharp/archive/refs/heads/main.zip" `
  -OutFile $sharpZip

Expand-Archive $sharpZip $BASE -Force
$SHARP_DIR = Get-ChildItem $BASE -Directory | Where-Object Name -Like "ml-sharp*"
Write-Host "SHARP_DIR: $BASE\$SHARP_DIR"

# ===============================
# 4. Creating a conda environment
# ===============================
$channels = @(
    "https://repo.anaconda.com/pkgs/main",
    "https://repo.anaconda.com/pkgs/r",
    "https://repo.anaconda.com/pkgs/msys2"
)

foreach ($channel in $channels) {
    Write-Host "Accepting ToS for: $channel"
    & $condaExe tos accept --override-channels --channel $channel 2>$null
}

& $condaExe create -y -n $ENVNAME python=$PYTHON

# ===============================
# 5. Requirements Installation
# ===============================

# Confirm the absolute path
$absPath = Resolve-Path "$BASE\$SHARP_DIR"
Write-Host "absPath: $absPath"

# Move to that location, including the drive (Push-Location also supports drive movement)
Push-Location $absPath

try {
    Write-Host "--- Installation Start ---"
    # Run `pip install -e .` directly (to prevent location mismatches)
    & $condaExe run -n $ENVNAME --no-capture-output pip install --no-cache-dir -e .
}
finally {
    Pop-Location
}

$sharpExe = "$BASE\Miniconda\envs\sharp\Scripts\sharp.exe"
Write-Host "sharpExe: $sharpExe"

# ===============================
# 6. Create a drag-and-drop compatible batch file
# ===============================
$BAT = "$BASE\run_sharp_dragdrop.bat"

@"
@echo off
setlocal

REM === Dragged Path ===
set INPUT=%~1

if "%INPUT%"=="" (
    echo Drag the folder containing the input images to this icon.
    pause
    exit /b
)

if not exist "%INPUT%\*" (
    echo Drag the folder containing the input images to this icon.
    pause
    exit /b
)

REM === Output Folder ===
set OUTPUT=%INPUT%\sharp_output

if not exist "%OUTPUT%" (
    mkdir "%OUTPUT%"
)

REM === SHARP Execution ===
"$sharpExe" predict -i "%INPUT%" -o "%OUTPUT%"

echo.
echo Processing is complete.
pause
"@ | Set-Content $BAT

# ---------- Shortcut ----------
$desktop = [Environment]::GetFolderPath("Desktop")
$lnk = "$desktop\SHARP (Drag & Drop).lnk"

$wsh = New-Object -ComObject WScript.Shell
$sc = $wsh.CreateShortcut($lnk)
$sc.TargetPath = $BAT
$sc.WorkingDirectory = $BASE
$sc.Save()

# ---------- Finished ----------
Write-Host ""
Write-Host "==============================="
Write-Host "Installation complete"
Write-Host "A shortcut named 'SHARP (Drag & Drop)'"
Write-Host "has been created on your desktop."
Write-Host "Drag the folder containing"
Write-Host "the input images to this icon."
Write-Host "==============================="

Write-Host ""
Write-Host "Press Enter to exit."
Read-Host
